# touch gesture management (touch.nvgt)
## Classes
### touch_gesture_manager

Detects and then converts raw touch screen finger data into gesture events which can be detected by your application or even converted to keyboard input.

touch_gesture_manager();

#### Remarks:

This is the highest level interface NVGT offers to accessing touch screens. It looks at what fingers are touching the screen and where, and by monitoring that data over time, derives gestures such as swipes and taps from that data before passing them along to whatever interface you register to receive them.

An interface is included by default (see the touch_keyboard_interface class), which convertes these events into simulated keyboard input for you so that adding touch support to your game becomes as simple as mapping gesture names to keycode lists.

The basic usage of this class is to create an instatnce of it somewhere, before attaching any number of touch_interface instances to the manager and finally calling manager.monitor() within the loops of your programs to receive gesture events.

#### Example:

```NVGT
#include"touch.nvgt"
#include"speech.nvgt"
touch_gesture_manager touch;
void main() {
	show_window("touch screen test");
	//attach multiple keys to a single gesture.
	int[] t = {KEY_TAB, KEY_LSHIFT};
	touch_keyboard_interface testsim(touch, {{"swipe_left1f", KEY_LEFT}, {"swipe_right1f", KEY_RIGHT}, {"swipe_up1f", KEY_UP}, {"swipe_down1f", KEY_DOWN}, {"double_tap1f", KEY_RETURN}, {"tripple_tap1f", t}, {"swipe_left2f", KEY_ESCAPE}, {"swipe_right2f", KEY_Q}, {"double_tap2f", KEY_SPACE}});
	touch.add_touch_interface(testsim);
	dictionary@d = {{"double_tap1f", KEY_X}, {"double_tap2f", KEY_Z}};
	// Demonstrate registering an interface for only a portion of the screen.
	touch.add_touch_interface(touch_keyboard_interface(touch, d, 0.5, 1.0, 0.5, 1.0));
	while (!key_pressed(KEY_ESCAPE) and !key_pressed(KEY_AC_BACK)) {
		wait(5);
		touch.monitor();
		if (key_pressed(KEY_LEFT)) speak("left");
		if (key_pressed(KEY_RIGHT)) speak("right");
		if (key_pressed(KEY_DOWN)) speak("down");
		if (key_pressed(KEY_UP)) speak("up");
		if (key_pressed(KEY_TAB)) {
			speak("tripple tap");
			if (key_down(KEY_LSHIFT) or key_down(KEY_RSHIFT)) speak("you are holding down shift", false);
		}
		if (key_pressed(KEY_SPACE)) speak("double tap 2 fingers");
		if (key_pressed(KEY_ESCAPE)) exit();
		if (key_pressed(KEY_Q)) speak("swipe right 2 fingers");
		if (key_pressed(KEY_RETURN)) speak("you just double tapped the screen!", false);
		if (key_pressed(KEY_X)) speak("You just double tapped another part of the screen!");
		if (key_pressed(KEY_Z)) speak("You just double tapped with 2 fingers on another part of the screen!");
	}
}
```



#### Methods
##### is_available

Determine whether any touch devices are available to receive events from.

bool is_available();

###### Returns:

bool: true if at least one touch device is available on the system, false otherwise.

###### Remarks:

It is worth noting that due to circumstances outside our control, sometimes touch devices don't appear in the list until after you have touched them at least once. Therefor you should not use this method at program startup to determine with finality that no touch device is available, but could instead use it during program lifetime to monitor for whether touch support appears or disappears.

###### Example:

```NVGT
touch_gesture_manager touch;
void main() {
	wait(1000); // Give the user some time to touch the screen to make sure it appears.
	if (!touch.is_available()) alert("warning", "This system does not appear to support touch screen devices");
}
```



##### monitor
Check for the latest touch events.

`void monitor();`

###### Remarks:
This function must be called in all of your game's loops in order for the system to work properly. Not doing this will lead to undefined behavior.


##### set_interfaces
Sets the list of interfaces that will receive touch events.

`bool set_touch_interfaces(touch_interface@[]@ interfaces, bool append = false);`

###### Arguments:
* `touch_interface@[]@ interfaces`: A list of interfaces that will receive touch events.
* `bool append = false`: Determines whether to append to the list of existing interfaces Vs. replacing it.

###### Returns:

bool: True if a change was made to the interface list, false otherwise.

###### Remarks:
You can pass multiple interfaces to a gesture manager because different interfaces can receive different events for various parts of the screen. An interface can simply return false in it's `is_bounds` method at any time to pass the gesture event to the next handler in the chain. Gesture interfaces are evaluated from newest to oldest.




### touch_keyboard_interface
Convert gesture events to simulated keyboard input.

`touch_keyboard_interface(touch_gesture_manager@parent, dictionary@map, float minx = TOUCH_UNCOORDINATED, float maxx = TOUCH_UNCOORDINATED, float miny = TOUCH_UNCOORDINATED, float maxy = TOUCH_UNCOORDINATED);`

#### Arguments:
* touch_gesture_manager@ parent: A handle to the manager you intend to add this interface to, parameter subject for removal in future.
* dictionary@ map: A mapping of gestures to keycode lists (see remarks).
* float minx, maxx, miny, maxy = TOUCH_UNCOORDINATED: The bounds of this interface, default for entire screen or custom.

#### Remarks:
This interface works by receiving a mapping of gesture names or IDs to lists of keycodes that should be simulated.

The basic format of gesture IDs consists of a gesture name followed by a number of fingers and the letter f. For example, to detect a 2 finger swipe right gesture, you would use the key "swipe_right2f" in the mapping dictionary. The available gesture names are as follows:
* swipe_left
* swipe_right
* swipe_up
* swipe_down
* double_tap
* tripple_tap





